---
title: PAC files · Cloudflare Zero Trust docs
description: You can apply Gateway HTTP and DNS policies at the browser level by
  configuring a Proxy Auto-Configuration (PAC) file to connect to a proxy
  endpoint.
lastUpdated: 2025-10-24T18:24:10.000Z
chatbotDeprioritize: false
source_url:
  html: https://developers.cloudflare.com/cloudflare-one/team-and-resources/devices/agentless/pac-files/
  md: https://developers.cloudflare.com/cloudflare-one/team-and-resources/devices/agentless/pac-files/index.md
---

Note

Only available on Enterprise plans.

You can apply Gateway HTTP and DNS policies at the browser level by configuring a Proxy Auto-Configuration (PAC) file to connect to a proxy endpoint.

A PAC file is a file containing a JavaScript function which can instruct a browser to forward traffic to a proxy server instead of directly to the destination server.

When end users visit a website, their browser will send the request to a Cloudflare proxy server associated with your account to be filtered by Gateway. Note that Gateway [cannot filter every type of HTTP traffic](#limitations) proxied using PAC files.

## Prerequisites

Install a [Cloudflare certificate](https://developers.cloudflare.com/cloudflare-one/team-and-resources/devices/user-side-certificates/) on your device.

## 1. Generate a proxy endpoint

You can generate a proxy endpoint in Cloudflare One or through the Cloudflare API.

Warning

All devices you add to the proxy endpoint will be able to access your Cloudflare Tunnel applications and services. If you only want to proxy web traffic, you can build a network policy that blocks those source IPs from connecting to your internal resources.

* Dashboard

  1. In [Cloudflare One](https://one.dash.cloudflare.com/), go to **Networks** > **Resolves & Proxies** > **Proxy endpoints**.

  2. Select **Create proxy endpoint**.

  3. Give your endpoint any name.

  4. Enter the public source IP address of your device(s) in CIDR notation. For example:

     * **IPv4**: `192.0.2.0/8`
     * **IPv6**: `2001:0db8:0000:0000:0000:1234:5678:0000/109`

     Note

     Gateway limits the prefix length of source networks for proxy endpoints to `/8` for IPv4 networks and `/32` for IPv6 networks.

  5. Select **Save endpoint** and confirm the endpoint creation.

  Your Cloudflare proxy server domain is of the form:

  ```txt
  https://<SUBDOMAIN>.proxy.cloudflare-gateway.com
  ```

* API

  1. [Create a proxy endpoint](https://developers.cloudflare.com/api/resources/zero_trust/subresources/gateway/subresources/proxy_endpoints/methods/create/) with the following call:

     Required API token permissions

     At least one of the following [token permissions](https://developers.cloudflare.com/fundamentals/api/reference/permissions/) is required:

     * `Zero Trust Write`

     ```bash
     curl "https://api.cloudflare.com/client/v4/accounts/$ACCOUNT_ID/gateway/proxy_endpoints" \
       --request POST \
       --header "Authorization: Bearer $CLOUDFLARE_API_TOKEN" \
       --json '{
         "name": "any_name",
         "ips": [
             "<PUBLIC_IP>",
             "<PUBLIC_IP2>",
             "<PUBLIC_IP3>"
         ]
       }'
     ```

     Replace `<PUBLIC_IP>` with the source IP address of your device in CIDR notation. For example:

     * **IPv4**: `192.0.2.0/8`
     * **IPv6**: `2001:0db8:0000:0000:0000:1234:5678:0000/32`

     Note

     Gateway limits the prefix length of source networks for proxy endpoints to `/8` for IPv4 networks and `/32` for IPv6 networks.

  2. After running the call, you should see an output similar to

     ```json
     {
       "result": {
         "id": "d969d7bf-ec28-4291-9af0-86825f472c21",
         "name": "test",
         "created_at": "2022-03-02T10:57:18.094789Z",
         "updated_at": "2022-03-02T10:57:18.094789Z",
         "ips": ["90.90.241.229/8"],
         "subdomain": "3ele0ss56t"
       },
       "success": true,
       "errors": [],
       "messages": []
     }
     ```

     Note the `subdomain` value returned by the API. Your Cloudflare proxy server domain is of the form:

     ```txt
     <SUBDOMAIN>.proxy.cloudflare-gateway.com
     ```

     In the example above, the subdomain is `3ele0ss56t` and the proxy server domain is `3ele0ss56t.proxy.cloudflare-gateway.com`.

## 2. Test your proxy server

1. In [Cloudflare One](https://one.dash.cloudflare.com/), create an [HTTP policy](https://developers.cloudflare.com/cloudflare-one/traffic-policies/http-policies/) for testing purposes. For example:

   | Selector | Operator | Value | Action |
   | - | - | - | - |
   | Domain | in | `example.com` | Block |

2. Verify that nothing is returned by a `curl` command:

   ```sh
   curl --ipv4 --proxytunnel --proxy https://3ele0ss56t.proxy.cloudflare-gateway.com https://example.com
   ```

If `curl` returns a `403` code, it means the public IP of your device does not match the one used to generate the proxy server. Make sure that WARP is turned off on your device and double-check that `curl` is not using IPv6 (use the `-4` option to force IPv4).

## 3. Create a PAC file

A PAC file is a text file that specifies which traffic should redirect to the proxy server.

Below is the default PAC file. You can [customize the file](https://developer.mozilla.org/en-US/docs/Web/HTTP/Proxy_servers_and_tunneling/Proxy_Auto-Configuration_PAC_file) and host it somewhere your browser can access, such as on an internal web server or on [Cloudflare Workers](https://developers.cloudflare.com/workers/).

```js
function FindProxyForURL(url, host) {
  // No proxy for private (RFC 1918) IP addresses (intranet sites)
  if (
    isInNet(dnsResolve(host), "10.0.0.0", "255.0.0.0") ||
    isInNet(dnsResolve(host), "172.16.0.0", "255.240.0.0") ||
    isInNet(dnsResolve(host), "192.168.0.0", "255.255.0.0")
  ) {
    return "DIRECT";
  }


  // No proxy for localhost
  if (isInNet(dnsResolve(host), "127.0.0.0", "255.0.0.0")) {
    return "DIRECT";
  }


  // Proxy all
  return "HTTPS 3ele0ss56t.proxy.cloudflare-gateway.com:443";
}
```

Note

* Make sure the directive used for the endpoint is `HTTPS` and not `PROXY`.
* You must use a PAC file instead of configuring the endpoint directly in the proxy configuration of the browser. This is because modern browsers still do not support HTTPS proxies without PAC files.
* Use a proper text editor such as VS Code to avoid added characters.

## 4. Configure your devices

All major browsers support PAC files. You can configure individual browsers, or you can configure system settings that apply to all browsers on the device. Multiple devices can call the same PAC file as long as their source IP addresses were included in the proxy endpoint configuration.

Chromium-based browsers

Chromium-based browsers (such as Google Chrome, Microsoft Edge, and Brave) rely on your operating system's proxy server settings. To configure your browser to use Gateway with PAC files, refer to the [macOS](https://support.apple.com/guide/mac-help/mchlp2591/mac) or [Windows](https://support.microsoft.com/windows/use-a-proxy-server-in-windows-03096c53-0554-4ffe-b6ab-8b1deee8dae1) documentation.

Mozilla Firefox

1. In Firefox, go to **Settings** and scroll down to **Network Settings**.
2. Select **Settings**.
3. Select **Automatic proxy configuration URL**.
4. Enter the URL where your PAC file is hosted, for example `https://proxy-pac.cflr.workers.dev/3ele0ss56t.pac`.
5. Select **OK**. HTTP traffic from Firefox is now being filtered by Gateway.

Safari

Safari relies on your operating system's proxy server settings. To configure your browser to use Gateway with PAC files, refer to the [macOS documentation](https://support.apple.com/guide/mac-help/mchlp2591/mac).

## 5. Test your HTTP policy

To test your configuration, you can test any [supported HTTP policy](#limitations), such as the example policy created in [Step 2](#2-test-your-proxy-server). When you go to `https://example.com` in your browser, you should see the Gateway block page.

You can now use the Proxy Endpoint selector in [network](https://developers.cloudflare.com/cloudflare-one/traffic-policies/network-policies/#proxy-endpoint) and [HTTP](https://developers.cloudflare.com/cloudflare-one/traffic-policies/http-policies/#proxy-endpoint) policies to filter traffic proxied via PAC files.

## Configure firewall

You may need to configure your organization's firewall to allow your users to connect to a proxy endpoint. Depending on your firewall, you will need to create a rule using either your proxy endpoint's domain or IP addresses.

To get the domain of a proxy endpoint:

* Dashboard

  1. In [Cloudflare One](https://one.dash.cloudflare.com/), go to **Networks** > **Resolves & Proxies** > **Proxy endpoints**.
  2. Choose the proxy endpoint. Select **Edit**.
  3. In **Proxy Endpoint**, copy the domain.

* API

  1. Use the [List proxy endpoints](https://developers.cloudflare.com/api/resources/zero_trust/subresources/gateway/subresources/proxy_endpoints/methods/list/) operation to get a list of your proxy endpoints and their details. For example:

     Required API token permissions

     At least one of the following [token permissions](https://developers.cloudflare.com/fundamentals/api/reference/permissions/) is required:

     * `Zero Trust Read`
     * `Zero Trust Write`

     ```bash
     curl "https://api.cloudflare.com/client/v4/accounts/$ACCOUNT_ID/gateway/proxy_endpoints" \
       --request GET \
       --header "Authorization: Bearer $CLOUDFLARE_API_TOKEN"
     ```

     ```json
     {
       "success": true,
       "result": {
         "id": "ed35569b41ce4d1facfe683550f54086",
         "created_at": "2014-01-01T05:20:00.12345Z",
         "ips": ["192.0.2.1/32"],
         "name": "DevOps team",
         "subdomain": "oli3n9zkz5.proxy.cloudflare-gateway.com",
         "updated_at": "2014-01-01T05:20:00.12345Z"
       }
     }
     ```

  2. Find the proxy endpoint you want to use.

  3. Copy the value of the `subdomain` key.

Using your proxy endpoint's domain, you can get the IP addresses assigned to the proxy endpoint:

* macOS and Linux

  1. Open a terminal.

  2. Run `dig` on your proxy endpoint's A records to get its IPv4 addresses. For example:

     ```bash
     dig A example.cloudflare-gateway.com +short
     ```

     ```txt
     162.159.36.5
     162.159.36.20
     ```

  3. Run `dig` on your proxy endpoint's AAAA records to get its IPv6 addresses. For example:

     ```bash
     dig AAAA example.cloudflare-gateway.com +short
     ```

     ```txt
     2606:4700:54::a29f:2407
     2606:4700:5c::a29f:2e07
     ```

* Windows

  1. Open a PowerShell terminal.

  2. Run `Resolve-DnsName` on your proxy endpoint's A records. Your proxy endpoint's IPv4 addresses will appear under `IPAddress`. For example:

     ```powershell
     Resolve-DnsName -Name example.cloudflare-gateway.com -Type A
     ```

     ```txt
     Name                                           Type   TTL   Section    IPAddress
     ----                                           ----   ---   -------    ---------
     example.cloudflare-gateway.com                 A      300   Answer     162.159.36.5
     example.cloudflare-gateway.com                 A      300   Answer     162.159.36.20
     ```

  3. Run `Resolve-DnsName` on your proxy endpoint's AAAA records. Your proxy endpoint's IPv6 addresses will appear under `IPAddress`. For example:

     ```powershell
     Resolve-DnsName -Name example.cloudflare-gateway.com -Type AAAA
     ```

     ```txt
     Name                                           Type   TTL   Section    IPAddress
     ----                                           ----   ---   -------    ---------
     example.cloudflare-gateway.com                 AAAA   300   Answer     2606:4700:5c::a29f:2e07
     example.cloudflare-gateway.com                 AAAA   300   Answer     2606:4700:54::a29f:2407
     ```

To ensure responses are allowed through your firewall, add an inbound rule to allow the static IPv4 address for Cloudflare proxy endpoints, `162.159.193.21`.

## Limitations

### Traffic limitations

The agentless HTTP proxy does not support [identity-based policies](https://developers.cloudflare.com/cloudflare-one/traffic-policies/identity-selectors/) or mTLS authentication.

To enforce HTTP policies for UDP traffic, you must turn on the [Gateway proxy for UDP](https://developers.cloudflare.com/cloudflare-one/traffic-policies/http-policies/http3/#enable-http3-inspection).

### Gateway DNS and resolver policies

Gateway DNS and resolver policies will always apply to traffic proxied via PAC files, regardless of device configuration.
